<?php
/*--------------------------------------------------------------------
 PriceBuilder.php 2020-2-17
 Gambio GmbH
 http://www.gambio.de
 Copyright (c) 2020 Gambio GmbH
 Released under the GNU General Public License (Version 2)
 [http://www.gnu.org/licenses/gpl-2.0.html]
 -------------------------------------------------------------------*/

declare(strict_types=1);

namespace Gambio\Shop\SellingUnit\Unit\Builders;

use Gambio\Shop\SellingUnit\Unit\Builders\Exceptions\UnfinishedBuildException;
use Gambio\Shop\SellingUnit\Unit\Builders\Interfaces\PriceBuilderInterface;

use Gambio\Shop\SellingUnit\Unit\Entities\Price;
use Gambio\Shop\SellingUnit\Unit\ValueObjects\PriceFormatted;
use Gambio\Shop\SellingUnit\Unit\ValueObjects\PricePlain;
use Gambio\Shop\SellingUnit\Unit\ValueObjects\PriceStatus;


/**
 * Class PriceBuilder
 * @package Gambio\Shop\SellingUnit\Unit\Builders
 */
class PriceBuilder implements PriceBuilderInterface
{
    /**
     * @var PricePlain
     */
    protected $pricePlain;
    
    /**
     * @var PriceFormatted
     */
    protected $priceFormatted = null;
    /**
     * @var int
     */
    protected $status;


    /**
     * @inheritDoc
     */
    public static function create(): PriceBuilderInterface
    {
        return new static;
    }
    
    
    /**
     * @inheritDoc
     */
    public function build(): Price
    {
        $fields = [
            'pricePlain',
            'priceFormatted',
            'status'
        ];
    
        foreach ($fields as $field) {
            
            if ($this->$field === null) {
                
                throw new UnfinishedBuildException($field . ' was not set');
            }
        }
        
        return new Price($this->pricePlain, $this->priceFormatted, $this->status);
    }
    
    
    /**
     * @inheritDoc
     */
    public function reset(): PriceBuilderInterface
    {
        $this->pricePlain = $this->priceFormatted = $this->status = null;
        
        return $this;
    }
    
    
    /**
     * @inheritDoc
     */
    public function withPricePlain(PricePlain $plain): PriceBuilderInterface
    {
        $this->pricePlain = $plain;
        
        return $this;
    }
    
    
    /**
     * @inheritDoc
     */
    public function withPriceFormatted(PriceFormatted $formatted): PriceBuilderInterface
    {
        $this->priceFormatted = $formatted;
        
        return $this;
    }

    /**
     * @inheritDoc
     */
    public function withStatus(PriceStatus $status): PriceBuilderInterface
    {
        $this->status = $status;
        return $this;
    }
}